<?php

if (!defined('ABSPATH')) exit;

class ACE_WebP_Admin
{
    private ACE_WebP_Converter $converter;

    public function __construct(ACE_WebP_Converter $converter)
    {
        $this->converter = $converter;

        add_action('admin_menu', [$this, 'add_menu']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
        add_action('wp_ajax_ace_webp_bulk_convert', [$this, 'ajax_bulk_convert']);
        add_action('wp_ajax_ace_webp_get_stats', [$this, 'ajax_get_stats']);
    }

    public function add_menu(): void
    {
        add_options_page(
            'WebP Optimizer',
            'WebP Optimizer',
            'manage_options',
            'ace-webp-optimizer',
            [$this, 'render_page']
        );
    }

    public function register_settings(): void
    {
        register_setting('ace_webp_settings', 'ace_webp_quality', [
            'type'              => 'integer',
            'sanitize_callback' => fn ($v) => max(1, min(100, (int) $v)),
            'default'           => 82,
        ]);
        register_setting('ace_webp_settings', 'ace_webp_auto_convert', [
            'type'              => 'integer',
            'sanitize_callback' => fn ($v) => $v ? 1 : 0,
            'default'           => 1,
        ]);
    }

    public function enqueue_assets(string $hook): void
    {
        if ($hook !== 'settings_page_ace-webp-optimizer') return;

        wp_enqueue_style(
            'ace-webp-admin',
            ACE_WEBP_URL . 'assets/admin.css',
            [],
            ACE_WEBP_VERSION
        );
        wp_enqueue_script(
            'ace-webp-admin',
            ACE_WEBP_URL . 'assets/admin.js',
            ['jquery'],
            ACE_WEBP_VERSION,
            true
        );
        wp_localize_script('ace-webp-admin', 'aceWebp', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce'   => wp_create_nonce('ace_webp_nonce'),
        ]);
    }

    /* ------------------------------------------------------------------
     * Settings page
     * ----------------------------------------------------------------*/
    public function render_page(): void
    {
        $quality      = (int) get_option('ace_webp_quality', 82);
        $auto_convert = (int) get_option('ace_webp_auto_convert', 1);
        $method       = $this->converter->get_method();
        $stats        = $this->get_stats();
        ?>
        <div class="wrap">
            <h1>ACE WebP Optimizer</h1>

            <div class="ace-webp-wrap">

                <!-- Settings Card -->
                <div class="ace-webp-card">
                    <h2>Settings</h2>
                    <form method="post" action="options.php">
                        <?php settings_fields('ace_webp_settings'); ?>

                        <table class="form-table">
                            <tr>
                                <th scope="row"><label for="ace_webp_quality">Quality</label></th>
                                <td>
                                    <div class="ace-webp-quality-wrap">
                                        <input type="range" id="ace_webp_quality" name="ace_webp_quality"
                                               min="1" max="100" value="<?php echo esc_attr($quality); ?>">
                                        <span class="ace-webp-quality-val" id="ace-webp-quality-display">
                                            <?php echo esc_html($quality); ?>
                                        </span>
                                    </div>
                                    <p class="description">1 = smallest file / lowest quality &mdash; 100 = largest file / highest quality. Recommended: 80&ndash;85.</p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">Auto-convert on upload</th>
                                <td>
                                    <label>
                                        <input type="checkbox" name="ace_webp_auto_convert" value="1"
                                            <?php checked($auto_convert, 1); ?>>
                                        Automatically convert new uploads to WebP
                                    </label>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">Conversion method</th>
                                <td><code><?php echo esc_html($method); ?></code></td>
                            </tr>
                        </table>

                        <?php submit_button(); ?>
                    </form>
                </div>

                <!-- Bulk Convert Card -->
                <div class="ace-webp-card">
                    <h2>Bulk Convert</h2>
                    <p>Scan every existing image in the Media Library and create WebP versions.</p>
                    <button type="button" class="button button-primary" id="ace-webp-bulk-convert">
                        Start Bulk Conversion
                    </button>
                    <div class="ace-webp-progress-wrap" id="ace-webp-progress-wrap">
                        <div class="ace-webp-progress-outer">
                            <div class="ace-webp-progress-bar" id="ace-webp-progress-bar"></div>
                        </div>
                        <p class="ace-webp-progress-text" id="ace-webp-progress-text">0 / 0 processed</p>
                    </div>
                </div>

                <!-- Statistics Card -->
                <div class="ace-webp-card">
                    <h2>Statistics</h2>
                    <div class="ace-webp-stats">
                        <div class="ace-webp-stat">
                            <span class="ace-webp-stat-num" id="ace-webp-stat-total">
                                <?php echo esc_html($stats['total']); ?>
                            </span>
                            <span class="ace-webp-stat-label">Total Images</span>
                        </div>
                        <div class="ace-webp-stat">
                            <span class="ace-webp-stat-num" id="ace-webp-stat-converted">
                                <?php echo esc_html($stats['converted']); ?>
                            </span>
                            <span class="ace-webp-stat-label">Converted</span>
                        </div>
                        <div class="ace-webp-stat">
                            <span class="ace-webp-stat-num" id="ace-webp-stat-saved">
                                <?php echo esc_html($stats['saved']); ?>
                            </span>
                            <span class="ace-webp-stat-label">MB Saved</span>
                        </div>
                    </div>
                </div>

            </div>
        </div>
        <?php
    }

    /* ------------------------------------------------------------------
     * AJAX: Bulk convert
     * ----------------------------------------------------------------*/
    public function ajax_bulk_convert(): void
    {
        check_ajax_referer('ace_webp_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }

        if (!$this->converter->is_supported()) {
            wp_send_json_error('WebP conversion not supported on this server.');
        }

        $offset     = (int) ($_POST['offset'] ?? 0);
        $batch_size = (int) ($_POST['batch_size'] ?? 20);
        $total      = $this->get_image_count();

        $attachments = get_posts([
            'post_type'      => 'attachment',
            'post_mime_type' => ['image/jpeg', 'image/png'],
            'posts_per_page' => $batch_size,
            'offset'         => $offset,
            'post_status'    => 'inherit',
            'fields'         => 'ids',
            'orderby'        => 'ID',
            'order'          => 'ASC',
        ]);

        $converted = 0;
        $quality   = (int) get_option('ace_webp_quality', 82);

        foreach ($attachments as $id) {
            $metadata = wp_get_attachment_metadata($id);
            if (!$metadata) continue;

            $file = get_attached_file($id);
            if (!$file || !file_exists($file)) continue;

            if ($this->converter->convert_file($file, $quality)) {
                $converted++;
            }

            if (!empty($metadata['sizes'])) {
                $dir = dirname($file);
                foreach ($metadata['sizes'] as $size) {
                    $this->converter->convert_file($dir . '/' . $size['file'], $quality);
                }
            }
        }

        wp_send_json_success([
            'total'     => $total,
            'converted' => $converted,
            'offset'    => $offset,
            'batch'     => $batch_size,
        ]);
    }

    /* ------------------------------------------------------------------
     * AJAX: Get statistics
     * ----------------------------------------------------------------*/
    public function ajax_get_stats(): void
    {
        check_ajax_referer('ace_webp_nonce', 'nonce');

        wp_send_json_success($this->get_stats());
    }

    /* ------------------------------------------------------------------
     * Helpers
     * ----------------------------------------------------------------*/
    private function get_image_count(): int
    {
        global $wpdb;

        return (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->posts}
             WHERE post_type = 'attachment'
               AND post_status = 'inherit'
               AND post_mime_type IN ('image/jpeg','image/png')"
        );
    }

    private function get_stats(): array
    {
        $total       = $this->get_image_count();
        $converted   = 0;
        $saved_bytes = 0;

        $attachments = get_posts([
            'post_type'      => 'attachment',
            'post_mime_type' => ['image/jpeg', 'image/png'],
            'posts_per_page' => -1,
            'post_status'    => 'inherit',
            'fields'         => 'ids',
        ]);

        foreach ($attachments as $id) {
            $file = get_attached_file($id);
            if (!$file || !file_exists($file)) continue;

            $webp = $file . '.webp';
            if (file_exists($webp)) {
                $converted++;
                $saved_bytes += filesize($file) - filesize($webp);
            }
        }

        return [
            'total'     => $total,
            'converted' => $converted,
            'saved'     => round($saved_bytes / 1048576, 2),
        ];
    }
}
